## Annotated Code
#Taken from https://github.com/cran/WRS2/
### Critical value calculation
#  The below code calculates the critical value for a one-way differences in medians test by bootstrapping over specified number of iterations (iter). n is a vector of sample sizes by the independent groups for which the one-way WR-ANOVA is being performed. The alpha value of 0.05 indicates that the test is done at a 95% confidence level. The results from each iteration are put together, and the lower 0.05% percentile is extracted as the critical value. Each iteration randomly samples a normal distribution. 

med1way.crit <- function(n,alpha=.05,iter=2000,TEST=NA,SEED=TRUE){
  #
  #  Determine the critical value for the function
  #  med1way, assuming normality, based on the sample
  #  sizes in n.
  #
  J<-length(n) # calculate the number of independent groups
  x<-list()
  w<-vector("numeric",J)
  xbar<-vector("numeric",J)
  if(SEED)set.seed(2)
  chk<-NA
  grp<-c(1:J)
  for (it in 1:iter){ # do this iter number of times. The number of iterations is set. 
    for(j in 1:J){
      x[[j]]<-rnorm(n[j]) # randomly sample from a normal distribition n[j] times, where n[j] is the number of observations in group j
      w[j]<-1/msmedse(x[[grp[j]]])^2 # Studentise the sample mean using method recommended by McKean and Shrader (1984). Ends up being 1/varience. msmedse is another function in the package.
      xbar[j]<-median(x[[grp[j]]]) # calculate the median of the n randomly sampled observations for each randomly sampled group
      n[j]<-length(x[[grp[j]]]) # i guess this is to check that the procesure has gone to plan and n[j] @ begining = n[j] @end
    }
    u<-sum(w) 
    xtil<-sum(w*xbar)/u 
    chk[it]<-sum(w*(xbar-xtil)^2)/(J-1) # test statistic calculation for the iteration
  }
  chk<-sort(chk)
  iv<-round((1-alpha)*iter)
  crit.val<-chk[iv] #grabs the critical value at the 95% percentile of the bootstrapped distribution
  pval<-NA
  if(is.na(TEST)){TEST = crit.val}  
  pval<-sum((TEST<=chk))/iter# used test statistic from med1way to calculate p-value
  list(crit.val=crit.val,p.value=pval) # output 
}


## WR-ANOVA
med1way <-
  function(formula, data, iter = 2000) {
    
    if (missing(data)) {
      mf <- model.frame(formula)
    } else {
      mf <- model.frame(formula, data)
    }
    cl <- match.call()
    
    alpha <- 0.05
    crit <- NA
    SEED <- TRUE
    x <- split(model.extract(mf, "response"), mf[,2])   
    
    grp <- 1:length(x)      
    J <- length(grp)  # The number of groups to be compared
    n <- vector("numeric",J)
    w <- vector("numeric",J)
    xbar <- vector("numeric",J)
    
    for(j in 1:J){
      xx <- !is.na(x[[j]]) 
      val <- x[[j]]
      x[[j]] <- val[xx]  # Remove missing values
      w[j] <- 1/msmedse(x[[grp[j]]], sewarn = FALSE)^2 
      xbar[j]<-median(x[[grp[j]]])
      n[j]<-length(x[[grp[j]]])
    }
    pval <- NA
    u <- sum(w)
    xtil <- sum(w*xbar)/u
    TEST <- sum(w*(xbar-xtil)^2)/(J-1) # calculate the test statistic based on the actual data
    
    if(is.na(crit)){
      temp <- med1way.crit(n,alpha,SEED=SEED,iter=iter,TEST=TEST) # compute med1way with test statistic
      crit.val <- temp$crit.val
    }
    if(!is.na(crit)) crit.val <- crit
    result <- list(test = TEST, crit.val = crit.val, p.value = temp$p.value, call = cl) #results report
    class(result) <- c("med1way")
    result
  }


### Post-hoc test
#Adapted from  https://github.com/cran/WRS2/blob/master/R/mcppb20.R
medPH <- function(formula, data, nboot = 2000){
  
  
  if (missing(data)) {
    mf <- model.frame(formula)
  } else {
    mf <- model.frame(formula, data)
  }
  cl <- match.call()
  
  x <- split(model.extract(mf, "response"), mf[,2])
  con=0
  alpha=.05
  grp=NA
  crit = med1way.crit(lengths(x), TEST = NA)$p.value # bootstrapped critical value
  
  con<-as.matrix(con)
  if(is.matrix(x)){
    xx<-list()
    for(i in 1:ncol(x)){
      xx[[i]]<-x[,i]
    }
    x<-xx
  }
  if(!is.list(x))stop("Data must be stored in list mode or in matrix mode.")
  if(!is.na(sum(grp))){  # Only analyze specified group pairs.
    xx<-list()
    for(i in 1:length(grp))xx[[i]]<-x[[grp[1]]]
    x<-xx
  }
  J<-length(x)
  tempn<-0
  for(j in 1:J){
    temp<-x[[j]]
    temp<-temp[!is.na(temp)] # Remove missing values.
    tempn[j]<-length(temp)
    x[[j]]<-temp
  }
  Jm<-J-1
  d<-ifelse(sum(con^2)==0,(J^2-J)/2,ncol(con))
  
  if(d> 10 && nboot <5000){
    warning("Suggest using nboot = 5000 when the number of contrasts exceeds 10.")
  }
  icl<-round(crit*nboot)+1 # upper confidence interval based on p-value
  icu<-round((1-crit)*nboot) # lower confidence interval based on p-value
  if(sum(con^2)==0){
    con<-matrix(0,J,d)
    id<-0
    for (j in 1:Jm){
      jp<-j+1
      for (k in jp:J){
        id<-id+1
        con[j,id]<-1
        con[k,id]<- -1
      }}}
  psihat<-matrix(0,ncol(con),7)
  dimnames(psihat)<-list(NULL,c("con.num","p_noboot","psihat","ci.lower",
                                "ci.upper","p-value_boot","p-val_corr"))
  bvec<-matrix(NA,nrow=J,ncol=nboot)
  #set.seed(2) # set seed of random number generator so that
  #             results can be duplicated.
  for(j in 1:J){
    #   paste("Working on group ",j)
    data<-matrix(sample(x[[j]],size=length(x[[j]])*nboot,replace=TRUE),nrow=nboot)
    bvec[j,]<-apply(data,1,median) # Bootstrapped trimmed means for jth group
  }
  test<-NA
  for (d in 1:ncol(con)){
    top<-0
    for (i in 1:J){
      top<-top+con[i,d]*bvec[i,]
    }
    test[d]<-(sum(top>0)+.5*sum(top==0))/nboot
    test[d]<-min(test[d],1-test[d])
    top<-sort(top)
    psihat[d,4]<-top[icl]
    psihat[d,5]<-top[icu]
  }
  for (d in 1:ncol(con)){
    psihat[d,1]<-d
    testit<-pb2gen(value~L1,melt(x[which(abs(con[,d])==1)]),est = "median")
    psihat[d,6]<-2*test[d]
    psihat[d,2]<-testit$p.value
    psihat[d,3]<-testit$test
  }
  psihat[,7] = p.adjust(psihat[,2], method = "hochberg")
  list(psihat=psihat,crit.p.value=2*crit,con=con)
  
  fnames <- as.character(names(x))
  
  groups <- t(apply(con, 2, function(cc) {
    c(which(cc == 1), which(cc == -1))
  }))
  
  psihat1 <- cbind(groups, psihat[, -c(1)])
  colnames(psihat1)[1:2] <- c("Group")
  
  result <- list(comp = psihat1, fnames = fnames, call = cl)
  class(result) <- "mcp1"
  result
  
}


